import React, { Fragment, useState } from "react";
import { Col, Container, Row, Form } from "react-bootstrap";
import { FieldValues, useForm } from "react-hook-form";
import { yupResolver } from "@hookform/resolvers/yup";
import * as yup from "yup";
import { postRequest } from "../common/API";
import variables from "../common/variables";
import { toast } from "react-toastify";
import { ContactDataTypes } from "../common/interfaces";

const validationSchema = yup
  .object({
    name: yup.string().required("Name is required"),
    company_name: yup.string().required("Company name is required"),
    email: yup
      .string()
      .email("Must be a valid email")
      .required("Email is required"),
  })
  .required();

const ContactFrom: React.FC = (props) => {
  const [loading, setLoading] = useState<boolean>(false);

  const {
    register,
    handleSubmit,
    formState: { errors },
    reset,
  } = useForm({
    resolver: yupResolver(validationSchema),
  });
  const onSubmit = async (data: FieldValues) => {
    setLoading(true);
    try {
      let postData: ContactDataTypes | any = {};
      postData.name = data.name;
      postData.company = data.company_name;
      postData.email = data.email;
      postData.phone = data.phone;
      postData.message = data.message;
      // send data to backend
      const res = await postRequest(variables.apiUrls.postContact, postData);
      if (res?.data.statusCode === variables.statusCodes.successCode) {
        toast.success(res.data.message);
        reset();
      } else {
        toast.error("Validation error. Please recheck your inputs!");
      }
      //reset form field

      setLoading(false);
    } catch (err) {
      toast.error("Something went wrong. Please try again!");
      setLoading(false);
    }
  };

  return (
    <Fragment>
      <Container className="contact_from">
        <Row className="px-3 px-lg-3 justify-content-between">
          <Col xl={3} lg={4} className="mb-md-4">
            <h4 className="title py-3">Drop us a line</h4>
            <p>
              Fill in this form or <strong>send us an e-mail</strong> with your
              inquiry.
            </p>
          </Col>
          <Col xl={8} lg={8}>
            <form onSubmit={handleSubmit(onSubmit)}>
              <Form.Group className="form_control_lg flex-wrap d-sm-flex">
                <Col
                  md={6}
                  sm={12}
                  className="from_control_inner contact_form_height"
                >
                  <Form.Label>Name*</Form.Label>
                  <Form.Control
                    className="input_item is-invalid"
                    type="text"
                    placeholder="Your name"
                    {...register("name")}
                    style={{
                      border: errors.name?.message
                        ? "1px solid #dc3545"
                        : "none",
                    }}
                  />
                  <div className="invalid-feedback f-12">
                    {errors.name?.message}
                  </div>
                </Col>

                <Col
                  md={6}
                  sm={12}
                  className="from_control_inner contact_form_height"
                >
                  <Form.Label>Company*</Form.Label>
                  <Form.Control
                    className="input_item is-invalid"
                    type="text"
                    placeholder="Your company name"
                    {...register("company_name")}
                    style={{
                      border: errors.company_name?.message
                        ? "1px solid #dc3545"
                        : "none",
                    }}
                  />
                  <div className="invalid-feedback f-12">
                    {errors.company_name?.message}
                  </div>
                </Col>

                <Col
                  md={6}
                  sm={12}
                  className="from_control_inner contact_form_height"
                >
                  <Form.Label>Email*</Form.Label>
                  <Form.Control
                    className="input_item is-invalid"
                    type="email"
                    placeholder="name@example.com"
                    style={{
                      border: errors.email?.message
                        ? "1px solid #dc3545"
                        : "none",
                    }}
                    {...register("email")}
                  />
                  <div className="invalid-feedback f-12">
                    {errors.email?.message}
                  </div>
                </Col>

                <Col
                  md={6}
                  sm={12}
                  className="from_control_inner contact_form_height"
                >
                  <Form.Label>Phone (optional)</Form.Label>
                  <Form.Control
                    className="input_item"
                    type="text"
                    placeholder="Your actual phone number"
                    {...register("phone")}
                  />
                </Col>
              </Form.Group>

              <Form.Group className="mb-4">
                <Form.Label>Message (optional)</Form.Label>
                <Form.Control
                  className="text_area_section"
                  as="textarea"
                  rows={5}
                  placeholder="Your message"
                  {...register("message")}
                />
              </Form.Group>
              <button className="btnStyle d-block d-lg-inline-block">
                {loading && (
                  <span
                    className="spinner-border spinner-border-sm"
                    role="status"
                    aria-hidden="true"
                    style={{ marginRight: "5px" }}
                  ></span>
                )}
                Send Message
              </button>
            </form>
          </Col>
        </Row>
      </Container>
    </Fragment>
  );
};

export default ContactFrom;
